function model = safaModel( mat, options, update)
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% PURPOSE: Depending upon the updating method, get the appropriate 
% calibration model
%--------------------------------------------------------------------------
%
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>


%% Get all models across all tuning parameters
%--------------------------------------------------------------------------
switch upper( update )
    
    case {'PPS'} 
        model = hfOrdinary( mat.Xp, mat.Yp, options.latent );
        
    case {'SPS','SSPS'}
        model = hfOrdinary( mat.Xs, mat.Ys, options.latent );
        
    case {'LMC','FA-1','FA-2'} 
        model = hfAugment( mat.Xp, mat.Yp, mat.Xs, mat.Ys, ...
                           options, update );
             
    case {'NARP','NARS','NARE','NARD'}   
        model = hfNAR( mat.Xp, mat.Yp, mat.R, options );  
                            
end  

%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end

function model = hfOrdinary( X, Y, k )
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Ordinary (PLS) regression on X and Y
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Mean-center
Xmean = mean(X,1);
Ymean = mean(Y,1);
X = bsxfun(@minus,X,Xmean);
Y = bsxfun(@minus,Y,Ymean);
% Regress
w = hfPLS(X,Y,k);
% Archive
model = struct('w',{w},'N',{size(w,2)},'Xmean',{Xmean},'Ymean',{Ymean});
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end



function model = hfAugment( Xp, Yp, Xs, Ys, options, update )
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Regress on augmented linear system where labeled secondary samples are 
% used in either a sample or feature augmented fashion.
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Mean-center primary data
meanXp = mean(Xp,1);
meanYp = mean(Yp,1);
Xpc    = bsxfun(@minus,Xp,meanXp);
Ypc    = bsxfun(@minus,Yp,meanYp);
% Mean-center seciondary data
meanXs = mean(Xs,1);
meanYs = mean(Ys,1);
Xsc    = bsxfun(@minus,Xs,meanXs);
Ysc    = bsxfun(@minus,Ys,meanYs);
% Augment and regress
W = cell(options.nlambda,1);
N = nan(options.nlambda,1);
update = upper(update);
for i = 1:options.nlambda
    % Augment
    lambda = options.lambda(i);
    switch update 
        case 'LMC'
            X = [Xpc;       lambda*Xsc];
            Y = [Ypc;       lambda*Ysc];
        case 'FA-1'
            X = [Xpc 0*Xpc; lambda*Xsc lambda*Xsc];
            Y = [Ypc;       lambda*Ysc];
        case 'FA-2'    
            X = [Xpc 0*Xpc; lambda*Xsc lambda*Xsc; 0*Xpc Xpc];
            Y = [Ypc;       lambda*Ysc;            zeros(size(Xpc,1),1)];  
    end    
    % Regress
    wi = hfPLS( X, Y, options.latent );
    W{i} = wi';
    N(i) = size(wi,2);
end
W = cell2mat(W)';
if ismember(update,{'FA-1','FA-2'})
    n = size(W,1); 
    dim = { 1:(n/2), (n/2+1):n };
    W = W(dim{1},:) + W(dim{2},:);
end    
% Archive
model = struct('w',{W},'N',{N(:)'},'Xmean',{meanXs},'Ymean',{meanYs});
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end


function model = hfNAR( Xp, Yp, R, options )
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Regress on augmented linear system where unlabeled secondary samples are 
% used in an augmented fashion via the R matrix.
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Mean-center data
Xmean = mean(Xp,1);
Ymean = mean(Yp,1);
Xpc = bsxfun(@minus,Xp,Xmean);
Ypc = bsxfun(@minus,Yp,Ymean);
%% Cycle through tau values
W = cell(options.nlambda,1);
N = nan(options.nlambda,1);
for i = 1:options.nlambda
    % Augment
    lambda = options.lambda(i);
    X = [Xpc; lambda*R];
    Y = [Ypc; zeros(size(R,1),1)];   
    % Regress
    wi = hfPLS( X, Y, options.latent );
    W{i} = wi';
    N(i) = size(wi,2);
end
W = cell2mat(W)';
% Archive
model = struct('w',{W},'N',{N(:)'},'Xmean',{Xmean},'Ymean',{Ymean});
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end



function w = hfPLS(X,Y,k)
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Wrapper for the BIDIAGONAL PARTIAL LEAST SQUARES (bPLS) routine.  
% - Checks that the number of latent vectors is appropriate.
% - bPLS.m solves ||X_k*b-y||_2 where X_k is the reconstruction of X using 
%   the first k dimensions of the Krylov subspace. 
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Compute rank
[m,n] = size(X);
r = min(m-1,n);
% Ensure the maximum number of latent vectors (k) does not exceed rank
k = k(:);
kmax = max(k);
nlatent = min(r,kmax);
% Regress
w = bPLS(X,Y,nlatent,[],1);
k( k > size(w,2) ) = [];
w = w(:,k);
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end

