function [B,U,R,V,F,norms]=bPLS(X,y,k,s,reorth)
%======================================================================
%
% Author: 		
%   Per Christian Hansen
%   Department of Informatics and Mathematical Modeling
%   Technical University of Denmark 
%   www2.imm.dtu.dk/~pch/
%
% Cosmetic tweaks for readability.  
%   Erik Andries
%   Center for Advanced Research Computing, 2009
% 	www.hpc.unm.edu/~andriese
% 
% For the PLS fanatics in chemometrics, this "PLS" implementation is
% akin to the "bidiagonal" PLS implementation as opposed to the
% conventional PLS implementation.
%
%====================================================================== 
%
% PURPOSE: 
% bPLS performs k steps of the LSQR Lanczos bidiagonalization 
% algorithm applied to the system: min || X*b - y ||.  The routine 
% returns all k solutions, stored as columns of the matrix B.  
% bPLS.m computes the filter factors associated with each step and 
% stores them column-wise in the matrix F.  Reorthogonalization 
% is controlled by means of modified Gram-Schmidt (MGS).  A fudge 
% threshold is used to prevent filter factors from exploding.
% 
% This routine computes the following decomposition of X
% (1)  X = U*R*V'
% where U and V are orthonormal and R is upper bidiagonal.
% The solution of min || X*b - y || at the i-th PLS factor uses the 
% pseudoinverse of Eq.(1) in the following fashion:
% (2)  b = V(:,1:i)*inv(R(1:i+1,1:i))*U(:,1:i+1)'*y.
% The matrices U, R and V are useful if one wants to compute 
% diagnostic merits such as leverages or spectral F-ratios.
%
% This is a standalone routine and does not require the Regularization
% Toolbox of [3].  To accomplish this, three functions---pythag.m,
% app_hh.m and gen_hh.m---are appended at the end of bPLS.m.
% 
%====================================================================== 
% 
% INPUT:
% [1] X: (m,n) matrix of data with the n-dimensional samples aligned
%        row-wise
% [2] y: (m,1) vector of response variables
% [3] k: positive integer indicating the number of PLS factors
% [4] s: (OPTIONAL) 
%     Singular values associated with the SVD decomposition.  If 
%     s is supplied, then the filter factors associated with the 
%     SVD basis are computed, otherwise they are not computed.  
% [5] reorth: (OPTIONAL) 
%     1 or 0.  1 indicates that the reorthonormalization of matrices 
%     U and V will be performed (recommended).  0 otherwise.
%
%====================================================================== 
% 
% OUTPUT:
% [1] B: (n,k) matrix of regression vectors such that B(:,i) is the 
%     regression vector associated with the i-th PLS factor.
% [2] U: (m,k+1) orthonormal matrix of left Lanczos vectors 
% [3] R: (k+1,k) bidiagonal matrix 
% [4] V: (n,k) orthonormal matrix of right Lanczos vectors.
% [5] F: (OPTIONAL) 
%     (ls,k) matrix of filter factors where F(:,i) are the PLS filter 
%     factors associated with the i-th PLS factor.  The variable s is 
%     the number of singular values supplied.  If s is supplied, then 
%     F will be computed, otherwise F will be empty, i.e. F=[].
% [6] norms: (OPTIONAL)
%     (k,2) matrix where norms(i,1) and norms(i,2) are the solution 
%     norm ||b|| and residual norm ||X*b-y|| for the i-th PLS factor. 

%
%====================================================================== 
% 
% USAGE:
% [B,U,R,V] = bPLS(X,y,k);
% [B,U,R,V,F,norms] = bPLS(X,y,k,s);
% [B,U,R,V,F,norms] = bPLS(X,y,k);
%
%====================================================================== 
% 
% REFERENCE: 
% [1] C. C. Paige & M. A. Saunders, "LSQR: an algorithm for
%     sparse linear equations and sparse least squares", 
%     ACM Trans. Math. Software 8 (1982), 43-71.
% [2] http://www2.imm.dtu.dk/~pch/Regutools/index.html
% [3] P. C. Hansen,  Regularization Tools: A Matlab package for 
%     analysis and solution of discrete ill-posed problems, 
%     Numerical Algorithms, 6 (1994), pp. 1-35.
%
%======================================================================

%----------------------------------------------------------------------
% Default inputs
%----------------------------------------------------------------------
if (nargin<4), s=[]; end
if isempty(s), GetFF=0; else GetFF=1; end
if (nargin<5), reorth=[]; end
if isempty(reorth), reorth=1; end
if (nargout<6), GetNorms=0; else GetNorms=1; end

%----------------------------------------------------------------------
% Check inputs
%---------------------------------------------------------------------- 
if all(y==0)
    error('Right-hand side must be nonzero.'); 
end
[m,n]=size(X);
my=length(y);
if ( m~=my ) 
    error('Number of rows of X must be equal to length of y.');  
end 
maxrank=min(m,n);
k=min(k,maxrank);
if (k<0) || (round(k)~=k) 
    error('k must be positive integer less than min(m,n).');  
end

%----------------------------------------------------------------------
% Fudge threshold to avoid exploding filter factors
%----------------------------------------------------------------------
fudge_thr = 1e-4;

%----------------------------------------------------------------------
% Initialization.
%----------------------------------------------------------------------
[m,n] = size(X); 
B = zeros(n,k);
U = zeros(m,k); 
V = zeros(n,k); 
R = zeros(k+1,k+1);
if (reorth==2)
	HHU = zeros(m,k); 	
	HHV = zeros(n,k);
	HHalpha = zeros(1,k); 
	HHbeta = HHalpha;
end	
eta = zeros(k,1);     % solution norm, i.e. ||b||
rho = eta;            % residual norm, i.e. ||X*b-y||
c2 = -1; 
s2 = 0; 
xnorm = 0; 
z = 0;
v = zeros(n,1); 
b = v; 
if GetFF
    ls = length(s);
    F = zeros(ls,k); 
    Fv = zeros(ls,1); 
    Fw = Fv;
    s = s.^2;
else
    F=[];
end	

%----------------------------------------------------------------------
% Compute the vector u 
%----------------------------------------------------------------------
beta = norm(y);
if ( reorth == 2 )
    [beta,HHbeta(1),HHU(:,1)] = gen_hh(y);
end
u = y/beta; 
U(:,1) = u;

%----------------------------------------------------------------------
% Compute the vector v
%----------------------------------------------------------------------
r = X'*u; 
alpha = norm(r);
if (reorth==2)
    [alpha,HHalpha(1),HHV(:,1)] = gen_hh(r);
end
v = r/alpha; 
V(:,1) = v;
R(1,1)=alpha;

%----------------------------------------------------------------------
% Intermediate results for filter factors
%----------------------------------------------------------------------
phi_bar = beta; 
rho_bar = alpha; 
w = v;
Fv = s/(alpha*beta); 
Fw = Fv;

%----------------------------------------------------------------------
% Perform Lanczos bidiagonalization iterations
%---------------------------------------------------------------------- 
for i=2:k+1

    alpha_old = alpha; 
    beta_old = beta;
	
    %------------------------------------------------------------------
    % Compute X*v - alpha*u and update beta and u.
	%------------------------------------------------------------------
    p = X*v - alpha*u;
	switch reorth
	    case 0 % No re-orthonormalization
			beta = norm(p); 
			u = p/beta;
	    case 1 % Re-orthonormalization via modified Gram Schmidt
			for j=1:i-1
				p = p - (U(:,j)'*p)*U(:,j); 
			end
			beta = norm(p); 
			u = p/beta;
		case 2 % Re-orthonormalization via Householder reflections
			for j=1:i-1
				p(j:m) = app_hh(p(j:m),HHbeta(j),HHU(j:m,j));
			end
			[beta,HHbeta(i),HHU(i:m,i)] = gen_hh(p(i:m));
			u = zeros(m,1); 
			u(i) = 1;
			for j=i:-1:1
				u(j:m) = app_hh(u(j:m),HHbeta(j),HHU(j:m,j));
			end
	end		
	
    %------------------------------------------------------------------
    % Compute X'*u - beta*v and update alpha and v.
	%------------------------------------------------------------------
    r = X'*u - beta*v;
	switch reorth
		case 0
			alpha=norm(r); 
			v = r/alpha;		
		case 1
			for j=1:i-1 
				r = r - (V(:,j)'*r)*V(:,j); 
			end
			alpha=norm(r); 
			v = r/alpha;
		case 2
			for j=1:i-1
				r(j:n) = app_hh(r(j:n),HHalpha(j),HHV(j:n,j));
			end
			[alpha,HHalpha(i),HHV(i:n,i)] = gen_hh(r(i:n));
			v = zeros(n,1); 
			v(i) = 1;
			for j=i:-1:1
				v(j:n) = app_hh(v(j:n),HHalpha(j),HHV(j:n,j));
			end
	end		

    %------------------------------------------------------------------
    % Update U, R and V.
	%------------------------------------------------------------------
    U(:,i)=u; 
    V(:,i)=v; 
	R(i,i-1)=beta;
    R(i,i)=alpha;
	
	%------------------------------------------------------------------
    % Construct and apply orthogonal transformation.
	%------------------------------------------------------------------
    rrho = pythag(rho_bar,beta); 
    c1 = rho_bar/rrho;
    s1 = beta/rrho; 
    theta = s1*alpha; 
    rho_bar = -c1*alpha;
    phi = c1*phi_bar; 
    phi_bar = s1*phi_bar;

	%------------------------------------------------------------------
    % Compute solution norm and residual norm if necessary;
	%------------------------------------------------------------------
    if GetNorms
        delta = s2*rrho; 
        gamma_bar = -c2*rrho; 
        rhs = phi - delta*z;
        z_bar = rhs/gamma_bar; 
        eta(i-1) = pythag(xnorm,z_bar);
        gamma = pythag(gamma_bar,theta);
        c2 = gamma_bar/gamma; 
        s2 = theta/gamma;
        z = rhs/gamma; 
        xnorm = pythag(xnorm,z);
        rho(i-1) = abs(phi_bar);
    end

	%------------------------------------------------------------------
    % Compute the filter factors.
	%------------------------------------------------------------------
    if GetFF
        if (i==2)
            Fv_old=Fv;
            Fv=Fv.*(s - beta^2 - alpha_old^2)/(alpha*beta);
            F(:,i-1)=(phi/rrho)*Fw;
        else
            tmp=Fv;
			tmp1=s - beta^2 - alpha_old^2;
			tmp2=Fv_old*alpha_old*beta_old;
			Fv=(Fv.*tmp1 - tmp2)/(alpha*beta);
			Fv_old=tmp;
			F(:,i-1)=F(:,i-2) + (phi/rrho)*Fw;
        end    
        if (i > 3)
		    condF1=( abs(F(:,i-2)-1) < fudge_thr );
			condF2=( abs(F(:,i-3)-1) < fudge_thr );
            f=find(condF1 & condF2);
            if (length(f) > 0) 
			    F(f,i-1) = ones(length(f),1); 
		    end
        end
        Fw=Fv - (theta/rrho)*Fw;    
    end
    
	%------------------------------------------------------------------
    % Update the solution.
	%------------------------------------------------------------------
    b=b + (phi/rrho)*w; 
    w=v - (theta/rrho)*w;
    B(:,i-1)=b;
end

%----------------------------------------------------------------------
% Trim the last column from R and V
%----------------------------------------------------------------------
R(:,end)=[];
V(:,end)=[];

%----------------------------------------------------------------------
% Store solution and residual norms if necessary
%----------------------------------------------------------------------
if GetNorms
    norms=[eta,rho];
end

%======================================================================
return

%======================================================================
%
% HELPER SUBFUNCTIONS FROM THE REGULARIZATION TOOLS
% a) pythag.m
% b) app_hh.m
% c) gen_hh.m
%
%======================================================================


function x = pythag(y,z)
%======================================================================
% PYTHAG Computes sqrt( y^2 + z^2 ).
% x = pythag(y,z)
% Returns sqrt(y^2 + z^2) but is careful to scale to avoid overflow.
% Christian H. Bischof, Argonne National Laboratory, 03/31/89.
%======================================================================
rmax = max(abs([y;z]));
if (rmax==0)
  x = 0;
else
  x = rmax*sqrt((y/rmax)^2 + (z/rmax)^2);
end
%======================================================================
return

function A = app_hh(A,beta,v)
%======================================================================
%APP_HH Apply a Householder transformation.
%
% A = app_hh(A,beta,v)
%
% Applies the Householder transformation, defined by
% vector v and scaler beta, to the matrix A; i.e.
%     A = (eye - beta*v*v')*A .
%
% Per Christian Hansen, IMM, 03/11/92.
%======================================================================
A = A - (beta*v)*(v'*A);
%======================================================================
return

function [x1,beta,v] = gen_hh(x)
%======================================================================
%GEN_HH Generate a Householder transformation.
%
% [x1,beta,v] = gen_hh(x)
%
% Given a vector x, gen_hh computes the scalar beta and the vector v
% determining a Householder transformation
%    H = (I - beta*v*v'),
% such that H*x = +-norm(x)*e_1. x1 is the first element of H*x.
%
% Per Christian Hansen, IMM, 11/11/1997.
%======================================================================
v = x; alpha = norm(v);
if (alpha==0),
  beta = 0;
else
  beta = 1/(alpha*(alpha + abs(v(1))));
end
if (v(1) >= 0)
  v(1) = v(1) + alpha; x1 = -alpha;
else
  v(1) = v(1) - alpha; x1 = +alpha;
end
%======================================================================
return
