function S = perfMetrics( yref, yhat )
%% COMPUTE STATISTICS ASSOCIATED WITH REFERENCE & PREDICTION SCATTERPLOT 
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% INPUT:
% [1] yref: [n x 1] vector of references  
% [2] yhat: [n x p] matrix of predictions across p models
%
%-------------------------------------------------------------------------- 
%
% OUTPUT:
% [1] S: Performance stat struct with the following fields
%     .rmse:    [1 x p] 
%     .slope:   [1 x p] slope of best-fit line
%     .r:       [1 x p] correlation coef. of best-fit line
%     .int      [1 x p] intercept of best-fot line
%
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
   

%% INITIALIZE OUTPUT
%--------------------------------------------------------------------------
p = size(yhat,2);
e = nan(1,p);
S = struct('rmse',{e},'slope',{e},'r2',{e},'int',{e});

%%  COMPUTE RMSE
%--------------------------------------------------------------------------
yerror = bsxfun(@minus,yhat,yref); 
S.rmse = sqrt( nanmean(yerror.^2,1) );

%% COMPUTE (YREF,YHAT) SCATTERPLOT STATS
%-------------------------------------------------------------------------- 
for i = 1:p
    % - Avoid NaN values in predictions
    idx = ~isnan(yhat(:,i));
    y1 = yref(idx);
    y2 = yhat(idx,i);
    % - Least squares fit 
    [S.slope(i),S.int(i),S.r2(i)] = lsFit(y1,y2);
end 

%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end


function [slope,intc,r2] = lsFit(yref,yhat)
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
% Compute R^2, slope and intercept associated with (reference,prediction)
% scatterplot
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
x = yref(:);
y = yhat(:);
if length(y) == 1
    slope = nan;
    intc  = nan;
    r2    = nan;
else
    if all(x==x(1))
        slope = nan;
        intc  = nan; 
        r2    = nan; 
    else    
        r     = (zscore(x)'*zscore(y)) / (length(x)-1);
        r2    = r.^2;
        coef  = [x ones(size(x))] \ y;    
        slope = coef(1);
        intc  = coef(2);
    end
end
%>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>
end


