function [group,aux]=yab(x,g,opt,xc)
% YAB: acronym for "Yet Another Boxplot"
%==========================================================================
% PURPOSE: 
% Create a simple, yet customizable, boxplot that does not require
% the MATLAB Statistics Toolbox.
% 
% INPUT:
% a) yab(x)
%    x is a vector or matrix.  If x is a vector, then only one boxplot
%    will be produced.  If x is a matrix, then one boxplot will be created
%    for each column of x.
%
% b) yab(x,g)
%    x and g are vectors of the same length. The argument g specifies 
%    one or more grouping variables, producing a separate boxplot for 
%    each set of x values sharing the same g value.  
%
% c) yab(x,g,opt);
%    The same as case b) except that the argument opt specifies how 
%    the quartiles are computed and how the plot handles are controlled 
%    (see yabOptions.m below for details). The fields in opt must conform 
%    to the fields detailed in yabOptions.m below. 
%
% d) yab(x,g,opt,xc)
%    The same as c) but 
% 
% OUTPUT:
% - group: n-array of structs where group(i) is a struct containing all of 
%   the summary statistics for the ith group
% - aux: struct containing auxiliary information.  Contains two fields
%   .opt: the options struct from yabOptions.m
%   .handle: struct containg the graphics handles used in the boxplot.
%
% USAGE: see yabDemo.m
% 
%==========================================================================

%
% Simple input check 
%
% - vectorize x
[mx,nx]=size(x);  
m=mx*nx;
x=x(:);
% - default options
if nargin<4, xc=[]; end
if nargin<3, opt=[]; end
if isempty(opt), opt=yabOptions; end    
% - default group ids
if nargin<2, g=[]; end; 
if (nx>1) && isempty(g), g=repmat(1:nx,mx,1); end  
if isempty(g), g=ones(m,1); end
g=g(:); 
if m~=length(g), error('x and g must be the same length'); end    
% - strip out NaNs 
nanidx=isnan(x);
x(nanidx)=[]; 
g(nanidx)=[]; 

%
% Compute statistics per group
%
[group,glabel] = yabStats(x,g,opt);  

%
% Generate the boxplot per group
%
handle=yabPlots(group,glabel,opt,xc);

%
% Archive auxiliary results.  
%
aux=struct('opt',{opt},'handle',{handle});

%==========================================================================
end




