function handle = yabPlots(group,glabel,opt,xc)
%==========================================================================
% INPUT: 
% yabPlots(group,glabel,opt)
% - group: n-vector of structs such that group(i) is the summary 
%   for the ith group (see yabStats.m for description of struct fields)
% - glabel: (n x 1) cell array such that glabel{i} is a string variable 
%   describing the the ith group.
% - opt: the options from yabOptions.m
% - xc: (OPTIONAL) n-vector indicating where the the ith boxplot should be
%   centered.  If not supplied, x(i)=i is the default.
% 
% OUTPUT:
% - handle: struct containing fields associated with the handle graphics of
%   the boxplot
%==========================================================================

%
% Create figure (if requested) and set figure properties 
%
if opt.newfigure
    figure;
    set(gcf,'color',[1 1 1]);
end
if opt.clf
    clf; 
end   

%
%
%
n=length(group);
if nargin<4, xc=[]; end
if isempty(xc), xc=1:n; end

%
% Create the boxplot for each group
%
n=length(group);
hold on;
%
cmap=opt.color.iqrfill;
if size(cmap,1)>1, colfix=false; else colfix=true; end
if ~colfix, opt.color.iqrfill=cmap(1,:); end    
handle.box=boxPlot(group(1),xc(1),opt);               
%
if n>1
    handle.box=repmat(handle.box,n,1);
    for i=2:n
        if ~colfix, opt.color.iqrfill=cmap(i,:); end  
        handle.box(i)=boxPlot(group(i),xc(i),opt);   
    end
end    
hold off;

%
% Set axes properties
%
% - x- and y-axis limits
xlim=[0.4 n+0.6];
ylim=get(gca,'ylim');
ylim=ylim+0.05*diff(ylim)*[-1 1];
set(gca,'color',[.7 .7 .7],'xlim',xlim,'ylim',ylim);
grid on;
% - group labeling if more than one group
if n==1
    set(gca,'xtick',1,'xticklabel',' ');
else    
    set(gca,'xtick',1:n,'xticklabel',glabel);
    %handle.xlabel=xlabel('GROUPS','fontweight','bold','fontsize',15);
end    
% - orient horizontally if requested
if ~opt.vertical
    view(90,90); 
end   
%==========================================================================
end



%==========================================================================
%==========================================================================
%==========================================================================
%========
%======== boxPlot.m
%========
%==========================================================================
%==========================================================================
%==========================================================================

function h=boxPlot(group,x,opt)
%==========================================================================

%
% Plot IQR bounding box
%
dx=opt.boxwidth/2;
xbox=[x-dx,x+dx,x+dx,x-dx,x-dx];
ybox=[group.Q1,group.Q1,group.Q3,group.Q3,group.Q1];
h.iqrfill=fill(xbox,ybox,opt.color.iqrfill);
h.iqrbox=plot(xbox,ybox);
set(h.iqrbox,'color',opt.color.iqrbox,'linewidth',opt.linewidth);

%
% Plot median line
%
h.median=plot([x-dx,x+dx],[group.Q2,group.Q2]);
set(h.median,'color',opt.color.median);
set(h.median,'linewidth',opt.linewidth);

%
% Plot the lines outside the IQR box
%
switch opt.whisker
    case 'datum', L=group.LW; U=group.UW;
    case 'fence', L=group.LF; U=group.UF;    
end    
Style = opt.outlinestyle;
h.outlines(1)=plot([x x],[L group.Q1],Style);
h.outlines(2)=plot([x x],[group.Q3 U],Style);

%
% Plot the whiskers
%
s=0.4; 
h.outlines(3)=plot([x-s*dx,x+s*dx],[L,L]);
h.outlines(4)=plot([x-s*dx,x+s*dx],[U,U]);
set(h.outlines,'color',opt.color.outlines);
set(h.outlines,'linewidth',opt.linewidth);

%
% Plot the outliers (if any)
%
i=0;
h.marker=[];  
h.markerSide=[]; 
if ~isempty(group.Lout)  % outside lower fence
    i=i+1;
    nout=length(group.Lout);
    xout=repmat(x,1,nout);
    h.marker(i)=plot(xout,group.Lout,opt.marker.type);    
    h.markerSide{i}='L';
end
if ~isempty(group.Uout)  % outside upper fence
    i=i+1;
    nout=length(group.Uout);
    xout=repmat(x,1,nout);
    h.marker(i)=plot(xout,group.Uout,opt.marker.type);    
    h.markerSide{i}='U';
end   
if ~isempty(h.marker)
    set(h.marker,'markeredgecolor',opt.marker.edgecolor,...
                 'markerfacecolor',opt.marker.facecolor,...
                 'markersize',opt.marker.size);
end
%==========================================================================
end

