function [group,glabel] = yabStats(x,g,opt)
%==========================================================================
% INPUT: 
% See tab.m for more detailed description of x, y and opt
% - x: m-vector of data
% - g: m-vector of group ids
% - opt: options from yabOptions.m
%
% OUTPUT:
% - group: n-vector of structs such that group(i) corresponds to the 
%   summary statistics for the ith group
% - glabel: (n x 1) cell array such that glabel{i} is a string variable 
%   describing the the ith group.
%==========================================================================

%
% Get unique elements of group and get labels for each group
%
[element,j1,gidx]=unique(g);
n=length(element);
if iscell(element)
    glabel=element; 
    g=gidx;                   
    element=unique(gidx);      
else
    glabel=cellstr(num2str(element)); 
end

%
% Get summary statistics per group
%
group=yabGroupStats(x,g,element(1),opt.method);
if n>1
    group=repmat(group,n,1);
    for i=2:n
        group(i)=yabGroupStats(x,g,element(i),opt.method);
    end
end  

%==========================================================================
end

%==========================================================================
%==========================================================================
%==========================================================================
%========
%======== yabGroupStats.m  
%======== 
%==========================================================================
%==========================================================================
%==========================================================================

function group=yabGroupStats(x,g,gi,method)
% Compute the summary statistics within each group of x
%==========================================================================
% x: data
% g: group vector 
% gi: unique element of g; gi indicates which group to key on
% method: string variable indicating how to compute the quartiles (see
%   yabQuartiles.m below for details)    
%==========================================================================

%
% label group information (group id and sample indices and data associated 
% with the the specific grooup id) 
%
group.id=gi;              
group.index=find(g==gi);
xi=sort(x( group.index ));
group.data=xi;

%
% 5-number summary of z
%
Q=yabQuartiles(xi,method);  
group.min=min(xi);
group.Q1=Q(1);
group.Q2=Q(2);
group.Q3=Q(3);
group.max=max(xi);

%
% Lower and upper fence
%
IQR = group.Q3-group.Q1;
group.LF=group.Q1 - 1.5*IQR;
group.UF=group.Q3 + 1.5*IQR;
if group.LF < group.min
    group.LF = group.min;
end
if group.UF > group.max
    group.UF = group.max;
end

%
% Lower and upper whiskers (data closest to fence but still within the 
% lower and upper fence)
%
group.LW=xi(find(xi >= group.LF-eps,1,'first'));
group.UW=xi(find(xi <= group.UF+eps,1,'last'));

%
% Determine the outliers (if any)
% 
group.Lout=[]; 
group.Uout=[]; 
if any(xi < group.LF), group.Lout=xi(xi < group.LF); end
if any(xi > group.UF), group.Uout=xi(xi > group.UF); end

%==========================================================================
end


%==========================================================================
%==========================================================================
%==========================================================================
%========
%======== yabQuartiles.m
%======== 
%==========================================================================
%==========================================================================
%==========================================================================



function Q=yabQuartiles(x,method)
% Compute the quartiles Q1, Q2 and Q3
%==========================================================================
% Compute the median Q2 in the usual way.  However, there are many ways to 
% compute the first and third quartiles.
% a) 'excluded': 
%    - Q1 is the median of lower half (excluding Q2)
%    - Q3 is the median of upper half (excluding Q2)
% b) 'included'
%    - Q1 is the median of lower half (including Q2)
%    - Q3 is the median of upper half (including Q2)
% c) 'weighted'
%    - Q1 is the weighted averaged of x(i) and x(i+1) where
%      i=floor(0.25*(n+1))
%    - Q3 is the weighted averaged of x(i) and x(i+1) where
%      i=floor(0.75*(n+1))
% d) 'matlab': how matlab computes quartiles (see prctile.m for details)
%==========================================================================

%
% Compute the median (Q2)
%
n=length(x);
Q=zeros(3,1);
[Q(2),i,even]=yabMedian(x);

% 
% Compute Q1 and Q3 depending on quartile method
%
switch method
    case 'excluded'
        Q(3)=yabMedian(x(i+1:n));
        if even
            Q(1)=yabMedian(x(1:i));   
        else    
            Q(1)=yabMedian(x(1:i-1)); 
        end    
    case 'included'
        Q(1)=yabMedian(x(1:i));
        if even
            Q(3)=yabMedian(x(i+1:n));
        else    
            Q(3)=yabMedian(x(i:n)); 
        end 
    case 'weighted'    
        % Q1
        j=(1/4)*(n+1); fj=floor(j);
        w=j-fj;
        Q(1)=(1-w)*x(fj) + w*x(fj+1);
        % Q3    
        j=(3/4)*(n+1); fj=floor(j);
        w=j-fj;
        Q(3)=(1-w)*x(fj) + w*x(fj+1);  
    case 'matlab'
        perc=100*(0.5:1:(n-0.5))/n;
        % Q1
        j=find(perc <= 25,1,'last');
        w=(25-perc(j))/(perc(j+1)-perc(j));
        Q(1)=(1-w)*x(j) + w*x(j+1);
        % Q3    
        j=find(perc <= 75,1,'last');
        w=(75-perc(j))/(perc(j+1)-perc(j));
        Q(3)=(1-w)*x(j) + w*x(j+1); 
end    
%==========================================================================
end


%==========================================================================
%==========================================================================
%==========================================================================
%========
%======== yabMedian.m 
%======== 
%==========================================================================
%==========================================================================
%==========================================================================



function [Q2,i,even]=yabMedian(x)
% Compute the median
%==========================================================================
% Return the 
% - median Q2
% - floor of the median position, i.e. i=floor((n+1)/2))
% - logical to indicate whether length of x is even or not
%==========================================================================
n=length(x);
i=floor((n+1)/2);
if rem(n,2)==0
    even=true;
    Q2=(x(i)+x(i+1))/2;
else
    even=false;
    Q2=x(i);
end    
%==========================================================================
end


